function [beta_hat] = Estimation_Phase(D, h, beta_hat0, pmin, pmax, delta, S)
% Calculates the maximum likelihood estimate of the parameters vector beta
%
% Inputs:
%   D: 2*t matrix of price-demand couples up to time period t.
%   h: Function handle for demand probability.
%   beta_hat0: Initial guess of the vector of parameters of the function h [beta0, beta1]
%   pmin, pmax: Minimum and maximum prices.
%   delta: For a chosen delta the price that maximizes the revenue is between
%   pmin+delta and pmax-delta.
%   S: Number of time periods in the selling season.
%
% Outputs:
%   beta_hat: Maximum likelihood estimate.

% Use fminsearch to maximize the function
options = optimset('MaxFunEvals',200*length(beta_hat0));
% Nelder-Mead simplex (direct search) method
[beta_hat, ~] = fminsearch(@(beta) -LogLikelihood_Function(D, h, beta), beta_hat0, options);

% what type of function is h?
h1 = @(x) exp(x)/(1+exp(x));
h2 = @(x) 1/(1+exp(-x));
h3 = @(x) exp(x);
h4 = @(x) x;

if strcmp(func2str(h), func2str(h1)) || strcmp(func2str(h), func2str(h2)) % h is the logit function
    
    if length(beta_hat0) == 2
        
        % Construct the set B
        bounds = @(x) [x(2) + 1/(pmin+delta)+1e-6;
            -x(2)-3;
            -x(1) + log(-x(2)*(pmin+delta)-1)-x(2)*(pmin+delta);
            x(1) - log(-x(2)*(pmax-delta)-1) + x(2)*(pmax-delta)];
        
        if any(bounds(beta_hat) > 0)
            % Project the vector onto the set B
            options = optimoptions('fmincon', 'Algorithm', 'interior-point', 'Display', 'off');
            
            beta_hat = fmincon(@(x) norm(x - beta_hat), beta_hat, [], [], [], [], [], [], @(x) deal(bounds(x),[]), options);
        end
        
    elseif length(beta_hat0) == 3
        
        bounds = @(x) [x(2) + 1/(pmin+delta) + 1e-6;
              -x(2)-5;
              -x(1) + log(-x(2)*(pmin+delta)-1) - x(2)*(pmin+delta) - x(3)*S;
              x(1) + x(3) - log(-x(2)*(pmax-delta)-1) + x(2)*(pmax-delta);
              x(3) + 1e-6;
              -x(3)-5];
        
        if any(bounds(beta_hat) > 0)
            % Project the vector onto the set B
            options = optimoptions('fmincon', 'Algorithm', 'interior-point', 'Display', 'off');
            
            beta_hat = fmincon(@(x) norm(x - beta_hat), beta_hat0, [], [], [], [], [], [], @(x) deal(bounds(x),[]), options);
        end
        
    end
    
    
elseif strcmp(func2str(h), func2str(h3)) % h is the exponential function
    
    if length(beta_hat) == 2
        
        % Construct the set B
        bounds = @(x) [x(2) + 1/(pmax-delta);
             -x(2)-1/(pmin+delta);
             x(1) + x(2)*(pmin+delta)+ 1e-6;
             - x(1) - 5];
        
        if any(bounds(beta_hat) > 0)
            % Project the vector onto the set B
            options = optimoptions('fmincon', 'Algorithm', 'interior-point', 'Display', 'off');
            
            beta_hat = fmincon(@(x) norm(x - beta_hat), beta_hat, [], [], [], [], [], [], @(x) deal(bounds(x),[]), options);
        end
        
    elseif length(beta_hat) == 3
        
        % Construct the set B
        bounds = @(x) [-1/(pmin+delta) -x(2);
            x(2) +1/(pmax-delta);
            x(1) -x(3) +1e-6;
            -5 -x(1) +1e-6;
            x(3) +1e-6;
            -5 -x(3)];
        
        if any(bounds(beta_hat) > 0)
            % Project the vector onto the set B
            options = optimoptions('fmincon', 'Algorithm', 'interior-point', 'Display', 'off');
            
            beta_hat = fmincon(@(x) norm(x - beta_hat), beta_hat, [], [], [], [], [], [], @(x) deal(bounds(x),[]), options);
        end
        
    end
    
elseif strcmp(func2str(h), func2str(h4)) % h is the linear function
    
    if length(beta_hat) == 2
        
        % Construct the set B
        bounds = @(x) [x(2) + x(1)/(2*(pmax-delta));
            - x(2) -  x(1)/(2*(pmin+delta));
            x(1) -2 + 1e-6;
            -x(1) + 1e-6];
        
        if any(bounds(beta_hat) > 0)
            % Project the vector onto the set B
            options = optimoptions('fmincon', 'Algorithm', 'interior-point', 'Display', 'off');
            
            beta_hat = fmincon(@(x) norm(x - beta_hat), beta_hat, [], [], [], [], [], [], @(x) deal(bounds(x),[]), options);
        end
        
    elseif length(beta_hat) == 3
        
        % Construct the set B
        bounds = @(x) [-(x(1)+x(3))/(2*(pmin+delta)) -x(2);
            x(2) + (x(1)+x(3)*S)/(2*(pmax-delta));
            -x(3)*S-x(1) +1e-6;
            x(1) -2 +x(3) +1e-6;
            -x(3) -5;
            x(3) +1e-6];
        
        if any(bounds(beta_hat) > 0)
            % Project the vector onto the set B
            options = optimoptions('fmincon', 'Algorithm', 'interior-point', 'Display', 'off');
            
            beta_hat = fmincon(@(x) norm(x - beta_hat), beta_hat, [], [], [], [], [], [], @(x) deal(bounds(x),[]), options);
        end                                             
        
    end
    
end

end

